/* --------------------------------------------------------------
  PreviewResizer.js 2019-06-07
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2019 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/


'use strict';

/**
 * Preview Resizer Widget
 *
 * This widget will handle the creation of the preview resize element of the StyleEdit frame. You
 * should use the "addPreviewType" method to add new preview types in the list.
 */
export default class PreviewResizer {
	/**
	 * Class Constructor 
	 * 
	 * @param {jQuery} $target Target selector of the widget.
	 * @param {TemplatePreview} templatePreview Instance of the template preview needed for changing the preview types.
	 */
	constructor($target, templatePreview) {
		StyleEdit.Validator.isObject($target);
		StyleEdit.Validator.isObject(templatePreview);
		
		/**
		 * @type {jQuery}
		 */
		this.$target = $target;
		
		/**
		 * @type {TemplatePreview}
		 */
		this.templatePreview = templatePreview;
		
		/**
		 * Contains the preview type instances
		 *
		 * @type {PreviewType[]}
		 */
		this.previewTypes = [];
	}
	
	/**
	 * Initialize the widget.
	 *
	 * Once the preview types added use this method to initialize the widget. It will fetch the
	 * correct Mustache templates and bind the event handlers.
	 */
	initialize() {
		this.templatePreview.initialize();
		
		// Create the available preview types list.
		const template = $('#preview-type-template').html();
		const html = Mustache.render(template, {previewTypes: this.previewTypes});
		
		this.$target.find('.dropdown-menu').html(html);
		
		this._bindEventHandlers();
		
		this.$target.find('.dropdown-menu li:last').trigger('click');
	}
	
	/**
	 * Add new preview type to the instance.
	 *
	 * @param {PreviewType} previewType Contains the preview information (width, height etc).
	 *
	 * @return {PreviewResizer} Returns the class instance for method chaining.
	 */
	addPreviewType(previewType) {
		StyleEdit.Validator.isObject(previewType);
		this.previewTypes.push(previewType);
		return this;
	}
	
	/**
	 * Bind the event handlers for the widget.
	 *
	 * @this PreviewResizer
	 *
	 * @private
	 */
	_bindEventHandlers() {
		this.$target.find('.dropdown-menu li')
			.off('click')
			.on('click', event => {
				const $target = $(event.currentTarget);
				
				$.each(this.previewTypes, (index, previewType) => {
					if ($target.hasClass(previewType.getClass())) {
						this.templatePreview.changePreview(previewType);
						this.$target.find('.dropdown-menu li.active').removeClass('active');
						$target.addClass('active');
						this.$target.find('.type-toggle i').attr('class', previewType.getIconClass() + '');
					}
				});
			});
	}
}
